/**************************************************************************************

Copyright (c) Hilscher Gesellschaft fuer Systemautomation mbH. All Rights Reserved.

***************************************************************************************

  $Id: AP_Urt.h 4490 2020-08-13 13:32:21Z Robert $:

  Description:
    USER - Headerfile of UART Driver Module

  Changes:
    Date        Description
    -----------------------------------------------------------------------------------
    2010-03-23  File created.

**************************************************************************************/


#ifndef __AP_URT_H
#define __AP_URT_H

#ifdef __cplusplus
extern "C" {
#endif
  
  /*
  **************************************************************************
  *   Size of one UART Control Block a Task has to allocate memory for
  **************************************************************************
  */ 
  #define DRV_UART_SIZE (112)
  /*
  **************************************************************************
  *   Definitions 
  **************************************************************************
  */ 
  typedef union RX_URT_HANDLER_UNtag {
     void FAR (CALLBACK FAR* fnIrq)(RX_HANDLE hInt, void FAR* pvPrm); /* Hardware Interrupt Service Routine */
     void FAR (CALLBACK FAR* fnPoll)(void FAR* pvInp); /* Software Timer Routine for polling mode */
  } RX_URT_HANDLER_UN;
 
  typedef void (CALLBACK FAR* APP_URT_RXRDY) (RX_HANDLE hUrt, void FAR* pvPrm, UINT uRxChar); 
  typedef void (CALLBACK FAR* APP_URT_TXEPTY)(RX_HANDLE hUrt, void FAR* pvPrm); 
  typedef void (CALLBACK FAR* APP_URT_ERR)   (RX_HANDLE hUrt, void FAR* pvPrm); 

  typedef struct RX_URT_CONFIG_Ttag {
    UINT                    uUrtNum;             /* UART number (0..2) */
    RX_UART_BAUDRATE        eBdRte;              /* baud rate configuration code */
    RX_UART_PARITY          ePrty;               /* parity configuration code */
    RX_UART_STOPBIT         eStp;                /* stop bit configuration code */
    RX_UART_DATABIT         eDat;                /* data bit configuration code */
    UINT                    uRxFifoLvl;          /* receive FIFO trigger level for real UARTs ([0..16], 0: FIFO disabled, recommended value: 12) */
    UINT                    uTxFifoLvl;          /* transmit FIFO trigger level for real UARTs ([0..16], 0: FIFO disabled, recommended value: 4) */

    /* UART */
    RX_UART_RTS             eRts;                /* RTS signal handling code */
    RX_UART_RTS_POLARITY    eRtsPol;             /* RTS signal polarity code */
    UINT                    uRtsForrun;          /* RTS signal forerun in AUTO mode */
    UINT                    uRtsTrail;           /* RTS signal trail in AUTO mode */
    RX_UART_CTS             eCts;                /* CTS signal support */
    RX_UART_CTS_POLARITY    eCtsPol;             /* CTS signal polarity */

    /* CDC/USB */
    UINT32                  ulFlags;             /* UART HAL / USB HAL behavior control flags, see RX_UART_FLAG_xxx */
    UINT16                  usUsbVendorID;       /* USB device vendor ID (VID) */
    UINT16                  usUsbProductID;      /* USB device product ID (PID) */
    UINT16                  usUsbProductRelease; /* USB device product release number */
    STRING                  szUsbVendorName[RX_UART_USB_VENDOR_NAME_LEN];
    STRING                  szUsbProductName[RX_UART_USB_PRODUCT_NAME_LEN];
    STRING                  szUsbSerialNumber[RX_UART_USB_SERIAL_NUMBER_LEN];
  } RX_URT_CONFIG_T;

  /*
  ************************************************************
  *   Function Prototypes
  ************************************************************
  */
  RX_FATAL FAR    DrvUrtInit              (const FAR void FAR*          pvCfg,
                                           UINT                         uNum); 

  RX_RESULT FAR   Drv_UrtCreateUart       (RX_HANDLE                    hUrt,
                                           const RX_UART_SET_T FAR*     ptCfg); 

  RX_RESULT FAR   Drv_UrtInitializeUart   (RX_HANDLE                    hUart,
                                           APP_URT_RXRDY                fnRxRdy,
                                           APP_URT_TXEPTY               fnTxEpty,
                                           APP_URT_ERR                  fnErr,
                                           void FAR*                    pvPrm,
                                           BOOLEAN                      fMod,
                                           RX_URT_HANDLER_UN FAR*       punUartHdl); 

  RX_RESULT FAR   Drv_UrtIdentifyUart     (const STRING FAR*            pszIdn,
                                           UINT                         uInst,
                                           RX_HANDLE FAR*               phUart); 

  RX_RESULT FAR   Drv_UrtEnableUart       (RX_HANDLE                    hUart); 

  RX_RESULT FAR   Drv_UrtDisableUart      (RX_HANDLE                    hUart); 
  
  RX_RESULT FAR   Drv_UrtEnableReceiver   (RX_HANDLE                    hUart,
                                           BOOLEAN                      fEnable);
  
  RX_RESULT FAR   Drv_UrtSetReceiveMode   (RX_HANDLE                    hUart);
  
  RX_RESULT FAR   Drv_UrtSendCharacter    (RX_HANDLE                    hUart,
                                           UINT8                        bChar); 
                                           
  RX_RESULT FAR   Drv_UrtSend             (RX_HANDLE                    hUart,
                                           const UINT8 FAR*             pabChars,
                                           UINT FAR*                    puLength);

  RX_RESULT FAR   Drv_UrtGetCharacter     (RX_HANDLE                    hUart,
                                           UINT8 FAR*                   pbChar);
                                           
  RX_RESULT FAR   Drv_UrtRecv             (RX_HANDLE                    hUart,
                                           UINT8 FAR*                   pabChars,
                                           UINT FAR*                    puLength);

  RX_RESULT FAR   Drv_UrtGetStatus        (RX_HANDLE    hUrt,
                                           BOOLEAN FAR* pfCts,
                                           BOOLEAN FAR* pfDsr,
                                           BOOLEAN FAR* pfDcd);

  RX_RESULT FAR   Drv_UrtReset            (RX_HANDLE                    hUart);

  RX_RESULT FAR   Drv_UrtEnableTransmitter(RX_HANDLE                    hUart,
                                           BOOLEAN                      fEnable); 
  
  RX_RESULT FAR   Drv_UrtSetTransmitMode  (RX_HANDLE                    hUart); 
  
  RX_RESULT FAR   Drv_UrtDeleteUart       (RX_HANDLE                    hUart);
  
  RX_RESULT FAR   Drv_UrtChangeRts        (RX_HANDLE                    hUart,
                                           BOOLEAN                      fRts); 

  RX_RESULT FAR   Drv_UrtGetCts           (RX_HANDLE                    hUart,
                                           BOOLEAN FAR*                 pfCts); 

  RX_RESULT FAR   Drv_UrtGetConfig        (RX_HANDLE                    hUart,
                                           RX_URT_CONFIG_T*             ptCfg);

  RX_RESULT FAR   Drv_UrtSetConfig        (RX_HANDLE                    hUart,
                                           RX_URT_CONFIG_T*             ptCfg);

  #define UART_ERROR_FRAMING_MSK  0x00000001
  #define UART_ERROR_PARITY_MSK   0x00000002
  #define UART_ERROR_BREAK_MSK    0x00000004
  #define UART_ERROR_OVERRUN_MSK  0x00000008

  RX_RESULT FAR   Drv_UrtGetLastError     (RX_HANDLE                    hUart,
                                           UINT8 FAR*                   pbChar,
                                           UINT  FAR*                   puiErrorFlags);


#ifdef __cplusplus
}
#endif

#endif
